<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\OrderHeader as Order;
use App\Models\OrderDetail;
use App\Models\Warehouse;
use App\Http\Requests\Orders\Index;
use App\Http\Requests\Orders\Show;
use App\Http\Requests\Orders\Create;
use App\Http\Requests\Orders\Store;
use App\Http\Requests\Orders\Edit;
use App\Http\Requests\Orders\Update;
use App\Http\Requests\Orders\UpdateManage;
use App\Http\Requests\Orders\Destroy;
use App\Models\Client;
use App\Models\GteBodega;
use App\Models\GteProducto;
use App\Models\GteProductosLinea;
use App\Models\OrderHistory;
use App\Models\Product;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Carbon\Carbon;

/**
 * Description of OrderController
 *
 * @author Tuhin Bepari <digitaldreams40@gmail.com>
 */

class OrderController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @param  Index  $request
     * @return \Illuminate\Http\Response
     */

    public function index(Index $request)
    {
        $search = $request->input('search');

        $orders = Order::orderBy('id', 'ASC')
        ->where('SalesPersonCode', Auth::user()->id)
        ->when($search, function ($query, $search) {
            return $query->where('CardCode', 'like', '%' . $search . '%')
                ->orWhere('CardName', 'like', '%' . $search . '%');
        })
        ->paginate(20);
        return view('pages.orders.index', ['records' => $orders]);
    }

    public function order_client(Index $request)
    {
        $search = $request->input('search');
        $date_start = $request->input('date_start');
        $date_end = $request->input('date_end');
        $status = $request->input('status');

        $clients = Order::orderBy('id', 'ASC')->paginate(20);
        $warehouses = Auth::user()->warehouses->pluck('BODEGA_ID');
        // return $warehouses;
        // $warehouses = ;
        $orders = Order::whereIn('warehouse_id', $warehouses)->where('status', '!=', 'Facturado')
        ->when($search, function ($query, $search) {
            return $query->where('CardCode', 'like', '%' . $search . '%')
                ->orWhere('CardName', 'like', '%' . $search . '%');
        })
        ->when($date_start, function ($query, $date_start) {
            return $query->whereDate('DocDate','>=' ,$date_start);
        })
        ->when($date_end, function ($query, $date_end) {
            return $query->whereDate('DocDate','<=' ,$date_end);
        })
        ->when($status, function ($query, $status) {
            return $query->where('status', $status);
        })
        ->paginate(15);
        // return $orders;
        return view('pages.orders.order_client', ['records' => $orders]);
    }

    public function order_client_history(Index $request)
    {
        $search = $request->input('search');
        $stage = $request->input('stage');

        $clients = Order::orderBy('id', 'ASC')->paginate(20);
        $warehouses = Auth::user()->warehouses->pluck('id');

        

        // return $warehouses;
        // $warehouses = ;
        $orders = OrderHistory::
        when($search, function ($query, $search) {
            return $query->where('CardCode', 'like', '%' . $search . '%')
                ->orWhere('CardName', 'like', '%' . $search . '%');
        })
        ->when($stage, function ($query, $stage) {
            $last_year = Carbon::now()->subYear();
            $last_six_months = Carbon::now()->subMonths(6);
            if ($stage == "Sin Recompra") {
                return $query->whereDate('DocDate','<', $last_year);
            } elseif ($stage == "Considerables") {
                return $query->whereDate('DocDate','>', $last_year)->whereDate('DocDate','<', $last_six_months);
            } elseif ($stage == "Reciente") {
                return $query->whereDate('DocDate','>', $last_six_months);
            }
        })
        ->distinct()
        ->orderBy("DocDate", "DESC")
        ->orderBy("CardName", "ASC")
        ->paginate(15, ['CardCode', 'CardName']);
        // return $orders;
        return view('pages.orders.order_client_history', ['records' => $orders]);
    }

    /**
    
     * Display the specified resource.
     *
     * @param  Show  $request
     * @param  Order  $order
     * @return \Illuminate\Http\Response
     */
    public function show(Show $request, Order $order)
    {

        return view('pages.orders.show', [
            'record' => $order,
        ]);
    }

    public function manage(Show $request, Order $order)
    {
        $order = Order::find($request->id);

        // return $order = 

        return view('pages.orders.manage', [
            'record' => $order,
        ]);
    }

    public function history_manage(Show $request, Order $order)
    {
        $orders = OrderHistory::where('CardCode', $request->CardCode)->orderBy("DocDate", "DESC")->get();

        // return $order = 

        // return $orders;

        return view('pages.orders.history_manage', [
            'orders' => $orders,
        ]);
    }
    /**
    
     * Show the form for creating a new resource.
     *
     * @param  Create  $request
     * @return \Illuminate\Http\Response
     */
    public function create(Create $request)
    {
        $client_code = $request->client_code;
        $client_name = $request->client_name;
        // $clients = Client::limit(10)->get();
        $products = Product::all();
        $types = Product::distinct('type')->get(['type']);
        $warehouses = GteBodega::all();
        //get user profiles
        $profiles = Auth::user()->getRoleNames();

        $source = "";

        if (in_array('multinivel', $profiles->toArray())) {
            $source = "Multinivel";
        }

        if (in_array('crm', $profiles->toArray()) || in_array('superadmin', $profiles->toArray())) {
            $source = "CRM";
        }

        return view('pages.orders.create', [
            'model' => new Order,
            'client_code' => $client_code,
            'client_name' => $client_name,
            'products' => $products,
            'types' => $types,
            'warehouses' => $warehouses,
            'new_client' => $request->new_client,
            'source' => $source,
        ]);
    }
    /**
     * Store a newly created resource in storage.
     *
     * @param  Store  $request
     * @return \Illuminate\Http\Response
     */

    public function create_history(Create $request)
    {

        $client_code = $request->client_code;
        $client_name = $request->client_name;
        $products = GteProducto::join('gte_lista_precios', 'gte_productos.PRODUCTO_ID', '=', 'gte_lista_precios.PRODUCTO_ID')
        ->join('gte_productos_lineas', 'gte_productos.LINEA_ID', '=', 'gte_productos_lineas.LINEA_ID')
        ->get();
        $types = GteProductosLinea::all();
        $warehouses = GteBodega::all();

        $user = Auth::user();


        return view('pages.orders.create_history', [
            'model' => new Order,
            'client_code' => $client_code,
            'client_name' => $client_name,
            'products' => $products,
            'types' => $types,
            'warehouses' => $warehouses,
            'new_client' => $request->new_client,
            'source' => $user->hasRole('multinivel')? "Multinivel" : "CRM",
        ]);
    }
    /**
     * Store a newly created resource in storage.
     *
     * @param  Store  $request
     * @return \Illuminate\Http\Response
     */

    public function storeorder(Request $request)
    {
        $tax = 0.15;

        $rules = [
            'CardCode'       => 'required|string',
            'warehouse_id'       => 'required|string',
            'details'   => 'present|array',
            'details.*.code' => 'required|string',
            'details.*.quantity' => 'required|numeric',
            'details.*.price' => 'required|numeric'
        ];

        $this->validate($request, $rules);

        if ($request->id){
            $order = Order::find($request->id);
            $order->CardCode = $request->CardCode;
            $order->CardName = $request->CardName;
            $order->Series = 1;
            $order->DocDate = date('Y-m-d');
            $order->DocDueDate = date('Y-m-d');
            $order->TaxDate = date('Y-m-d');
            $order->DocType = "C";
            $order->SalesPersonCode = Auth::user()->id;
            $order->DiscountPercent = $request->DiscountPercent;
            $order->warehouse_id = $request->warehouse_id;
            $order->user_id = Auth::user()->id;
            // $order->amount = $request->amount;
            $order->status = "Cotizado";
            $order->source = $request->source;
            $order->new_client = $request->new_client;
            // $order->spected_amount = $request->spected_amount;
            $order->save();
            
            OrderDetail::where('order_header_id', $order->id)->delete();
        }else{
            $order = new Order;
            $order->CardCode = $request->CardCode;
            $order->CardName = $request->CardName;
            $order->Series = 1;
            $order->DocDate = date('Y-m-d');
            $order->DocDueDate = date('Y-m-d');
            $order->TaxDate = date('Y-m-d');
            $order->DocType = "C";
            $order->SalesPersonCode = Auth::user()->id;
            $order->DiscountPercent = $request->DiscountPercent;
            $order->warehouse_id = $request->warehouse_id;
            $order->user_id = Auth::user()->id;
            // $order->amount = $request->amount;
            $order->status = "Cotizado";
            $order->source = $request->source;
            $order->new_client = $request->new_client;
            // $order->spected_amount = $request->spected_amount;
            $order->save();
        }


        $details = $request->details;
        $amountTotal = 0;
        $discountTotal = 0;
        $subtotalTotal = 0;
        $taxTotal = 0;
        $total = 0;
        foreach ($details as $detail) {
            $orderDetail = new OrderDetail;
            $orderDetail->order_header_id = $order->id;
            $orderDetail->ItemCode = $detail['code'];
            $orderDetail->Quantity = $detail['quantity'];
            $orderDetail->UnitPrice = $detail['price'];
            $orderDetail->DiscountPercent = $detail['discount'];
            // $orderDetail->WarehouseCode = $detail['warehouse'];
            // $orderDetail->TaxCode = $detail['tax'];
            $orderDetail->save();
            $amountTotal += $detail['quantity'] * $detail['price'];
            $discountTotal += $detail['quantity'] * $detail['price'] * $detail['discount'] / 100;
            $subtotal = $detail['quantity'] * $detail['price'] * (1 - $detail['discount'] / 100);
            $subtotalTotal += $subtotal;
            $taxTotal += $subtotal * $tax;
            $total += $subtotal + $subtotal * $tax;
        }
        $order->amount = $amountTotal;
        $order->discount = $discountTotal;
        $order->subtotal = $subtotalTotal;
        $order->tax = $taxTotal;
        $order->total = $total;
        $order->expected_amount = $subtotalTotal;
        $order->save();

        return redirect()->route('orders.index');

        //  return $request;
        //  $model=new Order;
        //  $model->fill($request->all());

        //  if ($model->save()) {

        //      session()->flash('app_message', 'Order saved successfully');
        //      return redirect()->route('products.index');
        //      } else {
        //          session()->flash('app_message', 'Something is wrong while saving Order');
        //      }
        //  return redirect()->back();
    }

    public function store(Store $request)
    {
        $model = new Order;
        $model->fill($request->all());

        if ($model->save()) {

            session()->flash('app_message', 'Order saved successfully');
            return redirect()->route('products.index');
        } else {
            session()->flash('app_message', 'Something is wrong while saving Order');
        }
        return redirect()->back();
    }
    /**
     * Show the form for editing the specified resource.
     *
     * @param  Edit  $request
     * @param  Order  $product
     * @return \Illuminate\Http\Response
     */
    public function edit(Edit $request, Order $order)
    {
        // return 'edit';
        $client_code = $request->client_code;
        $client_name = $request->client_name;
        $products = GteProducto::join('gte_lista_precios', 'gte_productos.PRODUCTO_ID', '=', 'gte_lista_precios.PRODUCTO_ID')
        ->join('gte_productos_lineas', 'gte_productos.LINEA_ID', '=', 'gte_productos_lineas.LINEA_ID')
        ->get();
        $types = GteProductosLinea::all();
        $warehouses = GteBodega::all();

        $user = Auth::user();

        return view('pages.orders.create_history', [
            'model' => $order,
            'client_code' => $client_code,
            'client_name' => $client_name,
            'products' => $products,
            'types' => $types,
            'warehouses' => $warehouses,
            'new_client' => $request->new_client,
            'source' => $user->hasRole('multinivel')? "Multinivel" : "CRM",
        ]);
    }
    /**
     * Update a existing resource in storage.
     *
     * @param  Update  $request
     * @param  Order  $product
     * @return \Illuminate\Http\Response
     */
    public function update(Update $request, Order $order)
    {
        $order->fill($request->all());

        if ($order->save()) {

            if ($order->status == "Facturado") {
                $this->sapCreateOrder($order);
            }

            session()->flash('app_message', 'Order successfully updated');
            return redirect()->route('products.index');
        } else {
            session()->flash('app_error', 'Something is wrong while updating Order');
        }
        return redirect()->back();
    }

    public function updatemanage(UpdateManage $request, Order $order)
    {
        $order = Order::find($request->id);
        $order->fill($request->all());

        if($order->status == "Facturado"){
            $this->completeOrder($request);
        }elseif($order->save()) {
            session()->flash('app_message', 'Order successfully updated');
            return redirect()->route('orders.order_client');
        } else {
            session()->flash('app_error', 'Something is wrong while updating Order');
        }
        return redirect()->back();
    }

    /**
     * Delete a  resource from  storage.
     *
     * @param  Destroy  $request
     * @param  Order  $order
     * @return \Illuminate\Http\Response
     * @throws \Exception
     */
    public function destroy(Destroy $request, Order $order)
    {
        if ($order->delete()) {
            session()->flash('app_message', 'Order successfully deleted');
        } else {
            session()->flash('app_error', 'Error occurred while deleting Order');
        }

        return redirect()->back();
    }

    public function productsbytype(Request $request)
    {
        // ad 0 to warehouse_id if is less than 10
        $warehouse_id = $request->warehouse_id < 10 ? "0" . $request->warehouse_id : (string)$request->warehouse_id;
        $products = GteProducto::join('gte_inventarios', 'gte_productos.PRODUCTO_ID', '=', 'gte_inventarios.PRODUCTO_ID')
        // ->join('gte_productos_lineas', 'gte_productos.PRODUCTO_ID', '=', 'gte_productos_lineas.PRODUCTO_ID')
        ->where('LINEA_ID', $request->type)
        ->where('BODEGA_ID', $warehouse_id)
        ->get();

        $makes = Product::distinct('make')->where('type', $request->type)->get(['make']);
        return ["makes" => $makes, "products" => $products];
    }

    public function productsbysearch(Request $request)
    {
        // ad 0 to warehouse_id if is less than 10
        $warehouse_id = $request->warehouse_id < 10 ? "0" . $request->warehouse_id : (string)$request->warehouse_id;
        $products = GteProducto::join('gte_inventarios', 'gte_productos.PRODUCTO_ID', '=', 'gte_inventarios.PRODUCTO_ID')
        // ->join('gte_productos_lineas', 'gte_productos.PRODUCTO_ID', '=', 'gte_productos_lineas.PRODUCTO_ID')
        // ->where('LINEA_ID', $request->type)
        ->where('BODEGA_ID', $warehouse_id)
        ->when($request->search, function ($query, $search) {
            return $query->where('gte_productos.NOMBRE_PRODUCTO', 'like', '%' . $search . '%');
        })
        ->get();

        $makes = Product::distinct('make')->where('type', $request->type)->get(['make']);
        return ["makes" => $makes, "products" => $products];
    }

    private function sapLogin($client)
    {
        $SAP_LOGIN_URL = env("SAP_LOGIN_URL", "https://181.39.41.146:1007/b1s/v1/Login");
        $SAP_COMPANYDB = env("SAP_COMPANYDB", "SBO_NTE_PRUEBAS");
        $SAP_USERNAME = env("SAP_USERNAME", "manager");
        $SAP_PASSWORD = env("SAP_PASSWORD", "E2tc$4@");

        // $client = new \GuzzleHttp\Client(['verify' => false ]);

        $headers = [
            "Content-Type" => "application/json"
        ];

        $body = [
            "CompanyDB" => $SAP_COMPANYDB,
            "UserName" => $SAP_USERNAME,
            "Password" => $SAP_PASSWORD
        ];

        $response = $client->post($SAP_LOGIN_URL, [
            "headers" => $headers,
            "json" => $body,
            "debug" => true
        ]);

        $body = $response->getBody();

        $status = $response->getStatusCode();

        if ($status == 200) {
            $session = json_decode($body);
            Log::info(json_encode($session));
            Log::info("SAP session opened: " . $session->SessionId);
            // Log::info("SAP cookies: ".$response->getHeader("Set-Cookie");
            return $session->SessionId;
        } else {
            Log::error("Error opening SAP session: " . $body);
            return null;
        }
    }

    private function sapLogout($client)
    {
        $SAP_LOGOUT_URL = env("SAP_LOGIN_URL", "https://181.39.41.146:1007/b1s/v1/Logout");

        // $client = new \GuzzleHttp\Client(['verify' => false]);

        $headers = [
            "Content-Type" => "application/json"
        ];

        $response = $client->post($SAP_LOGOUT_URL, [
            "headers" => $headers
        ]);

        $body = $response->getBody();

        $status = $response->getStatusCode();

        if ($status == 204) {
            Log::info("SAP session closed");
            return true;
        } else {
            Log::error("Error closing SAP session: " . $body);
            return null;
        }
    }

    private function sapCreateOrder($order)
    {
        $SAP_CREATE_ORDER_URL = env("SAP_CREATE_ORDER_URL", "https://181.39.41.146:1007/b1s/v1/Orders");

        $client = new \GuzzleHttp\Client(['verify' => false, 'cookies' => true]);
        $sessionId = $this->sapLogin($client);

        if ($order->new_client) {
            $newClient = Client::where("CardCode", $order->CardCode)->first();
            if($newClient->created == 0){
                $this->createSapClient($client,$sessionId, $newClient);
            }
        }

        $bodega = GteBodega::where('BODEGA_ID', $order->warehouse_id)->first();
        $serie = $bodega->sucursal->serie->serie;
        Log::info("Serie: " . $serie);

        $body = [
            "Series" => $serie,
            "DocDate" => date('Y-m-d'),
            "TaxDate" => date('Y-m-d'),
            "DocDueDate" => date('Y-m-d'),
            "CardCode" => "CN1758774978",
            "SalesPersonCode" => 1, //TODO cambiar por el usuario 287
            "Comments" => $order->Comments,
            "DiscountPercent" => $order->DiscountPercent,
            "DocType" => $order->DocType,
            "BPL_IDAssignedToInvoice" => $order->BPL_IDAssignedToInvoice,
            "DocumentLines" => []
        ];

        $details = $order->orderDetails;
        foreach ($details as $detail) {
            $line = [
                "ItemCode" => $detail->ItemCode,
                "Quantity" => $detail->Quantity,
                "UnitPrice" => $detail->UnitPrice,
                "DiscountPercent" => $detail->DiscountPercent,
                "TaxCode" => "IVAVS15", //verificar con el nuevo iva
                "WarehouseCode" => $order->warehouse_id < 10 ? "0" . $order->warehouse_id : $order->warehouse_id
            ];
            array_push($body["DocumentLines"], $line);
        };

        Log::info("Body: " . json_encode($body));

        $headers = [
            "SessionId" => $sessionId,
            "Content-Type" => "application/json"
        ];

        $response = $client->post($SAP_CREATE_ORDER_URL, [
            "headers" => $headers,
            "json" => $body
        ]);
    
        $body = $response->getBody();

        $status = $response->getStatusCode();

        if ($status == 201) {
            $createdOrder = json_decode($body);

            $order->DocEntry = $createdOrder->DocEntry;
            $order->DocNum = $createdOrder->DocNum;

            $order->save();
            // return $session->SessionId;
            session()->flash('sap_message', 'Order successfully created in SAP');
            Log::info("Order created in SAP: " . $order->DocEntry);
        } else {
            Log::error("Error creating order in SAP: " . json_encode($body));
            $error = json_decode($body);
            Log::error("Error creating2 order in SAP: " . $error->error->message->value);
            session()->flash('sap_message_error', $error->error->message->value);
            // return null;
        }

        $this->sapLogout($client);

    }

    private function createSapClient($client, $sessionId, Client $newClient)
    {
        $SAP_CREATE_CLIENT_URL = env("SAP_CREATE_CLIENT_URL", "https://181.39.41.146:1007/b1s/v1/BusinessPartners");

        $headers = [
            "Content-Type" => "application/json",
            "SessionId" => $sessionId
        ];

        $body = [
            "CardCode" => $newClient->CardCode,
            "CardName" => $newClient->CardName,
            "CardType" => $newClient->CardType,
            "GroupCode" => $newClient->GroupCode,
            "CompanyPrivate" => $newClient->CmpPrivate,
            "FederalTaxID" => substr(strval($newClient->CardCode),1),
            "Phone1" => $newClient->Phone1,
            "Cellular" => $newClient->Cellular,
            "EmailAddress" => $newClient->E_Mail,
            "PayTermsGrpCode" => -1,
            "PriceListNum" => 1,
            "SalesPersonCode" => -1,
            "U_HT_TIPO_ID" => $newClient->U_HT_TIPO_ID,
            "U_HBT_TIPO_ID" => "02",
            "U_HBT_TIPPRO" => "01",
            "U_HBT_TIPCONT" => null,
            "U_HBT_PARTREL" => null,
            "U_HT_DAS_ENTRET" => "02",
            "U_HT_DAS_CATEGO" => null,
            "U_HT_DAS_SUBGRU" => "06",
            "U_HT_DAS_GESCOB" => null,
            "U_HBT_SEXO" => "F",
            "U_HBT_ESTCIV" => "S",
            "U_HT_DAS_CREPRO" => null,
            "U_HT_DAS_GRAIVA" => null,
            "U_HT_DAS_FORPAG" => null,
            "U_HT_DAS_CONDIC" => "03",
            "U_HT_DAS_CONBLO" => "0",
            "U_HBT_TIPO_PAGO" => "01",
            "U_HBT_DBL_TRIB" => "NO",
            "U_HT_FPAGO" => "01",
            "U_HBT_CLASUJ" => "N",
            "U_HBT_PAIS_PAGO" => "EC",
            "U_HBT_ORIING" => "I",
            "U_HBT_PAG_RET" => "SI",
            "BPAddresses" => [
                [
                    "AddressName" => "FACTURACION",
                    "AddressType" => "B",
                    "Street" => $newClient->Address,
                    "City" => $newClient->City,
                    "State" => $newClient->State,
                    "Country" => "EC",
                    "Block" => $newClient->Block
                ]
            ]
        ];

        // Log::info("Body: " . json_encode($body));

        //code...
        $response = $client->post($SAP_CREATE_CLIENT_URL, [
            "headers" => $headers,
            "json" => $body
        ]);

        $body = $response->getBody();

        $status = $response->getStatusCode();

        if ($status == 201) {
            Log::info("Client created in SAP: " . $body);
            //update client
            $client = Client::where("CardCode", $newClient->CardCode)->first();
            $client->created = 1;
            $client->save();
            session()->flash('sap_message', 'Client successfully created in SAP');
        } else {
            // return null;
            session()->flash('sap_message_error', 'Error creating client in SAP');
            Log::error("Error creating client in SAP: " . $body);
        }

    }

    public function completeOrder(Request $request)
    {
        $order = Order::find($request->id);
        try {
            //code...
            $this->sapCreateOrder($order);
            $order->status = "Facturado";
            $order->save();
        } catch (\GuzzleHttp\Exception\ClientException $e) {
            //throw $th;
            $error = json_decode($e->getResponse()->getBody());
            session()->flash('sap_message_error', $error->error->message->value);
            Log::error("Error creating order in SAP: " . $e->getResponse()->getBody());
        }
       

        
    }
}
